"use client";
import { useState } from "react";
import Link from "next/link";
import { ArrowLeft, Eye, EyeOff, AlertCircle, ShieldQuestion } from "lucide-react";
import { cn } from "@/lib/utils";
import { validateForgotPass } from "@/lib/auth-validation";

export default function ForgotPassword() {
  const [step, setStep] = useState(1); // 1: Identifier, 2: OTP, 3: Reset
  const [showPass, setShowPass] = useState(false);
  const [error, setError] = useState("");
  const [formData, setFormData] = useState({
    identifier: "",
    otp: "",
    password: "",
    confirmPassword: ""
  });

  const handleNext = (e: React.FormEvent) => {
    e.preventDefault();
    const validationError = validateForgotPass(step, formData);
    if (validationError) {
      setError(validationError);
      return;
    }
    setError("");
    if (step < 3) setStep(step + 1);
    else console.log("Password Reset Success");
  };

  return (
    <div className="min-h-screen bg-background flex flex-col items-center justify-center p-4 w-full">
      <div className="w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] border border-border shadow-2xl overflow-hidden flex flex-col md:flex-row min-h-[500px]">
        
        {/* Left Side: Contextual Info */}
        <div className="flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
          <div className="space-y-6">
            <div className="text-2xl font-bold tracking-tighter">
                <span className="text-[#4285F4]">I</span><span className="text-[#EA4335]">n</span><span className="text-[#FBBC05]">d</span><span className="text-[#34A853]">z</span><span className="text-[#4285F4]">s</span>
            </div>
            <div className="space-y-2">
              <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                {step === 1 && "Account recovery"}
                {step === 2 && "Check your phone"}
                {step === 3 && "Change password"}
              </h1>
              <p className="text-muted-foreground text-lg font-light max-w-sm">
                {step === 1 && "To help keep your account secure, Indzs needs to confirm this account belongs to you."}
                {step === 2 && `Enter the verification code sent to your device.`}
                {step === 3 && "Create a new, strong password that you don't use for other websites."}
              </p>
            </div>
          </div>
          {step > 1 && (
            <button onClick={() => setStep(step - 1)} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all">
              <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
            </button>
          )}
        </div>

        {/* Right Side: Inputs */}
        <div className="flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">
          <form onSubmit={handleNext} className="space-y-6 animate-in fade-in slide-in-from-right-4 duration-500">
            
            {/* Step 1: Identity */}
            {step === 1 && (
              <div className="relative group">
                <input
                  type="text" required
                  className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 text-foreground outline-none transition-all peer placeholder-transparent", error && "border-destructive")}
                  placeholder="Email or phone"
                  value={formData.identifier}
                  onChange={(e) => setFormData({...formData, identifier: e.target.value})}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all pointer-events-none peer-focus:-top-3 peer-focus:left-3 peer-focus:text-sm peer-focus:text-sidebar-primary peer-focus:bg-card peer-focus:px-2 peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:left-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card peer-[:not(:placeholder-shown)]:px-2">
                  Email or phone
                </label>
              </div>
            )}

            {/* Step 2: Verification Code */}
            {step === 2 && (
               <div className="relative">
               <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">G -</span>
               <input
                   type="text" maxLength={6} autoFocus
                   className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-12 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none", error && "border-destructive")}
                   value={formData.otp}
                   onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })}
               />
           </div>
            )}

            {/* Step 3: New Password */}
            {step === 3 && (
              <div className="space-y-4">
                <div className="relative">
                  <input
                    type={showPass ? "text" : "password"} required
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                    placeholder="New password"
                    onChange={(e) => setFormData({...formData, password: e.target.value})}
                  />
                  <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm px-1">New password</label>
                </div>
                <div className="relative">
                  <input
                    type={showPass ? "text" : "password"} required
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                    placeholder="Confirm"
                    onChange={(e) => setFormData({...formData, confirmPassword: e.target.value})}
                  />
                  <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm px-1">Confirm</label>
                  <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">
                      {showPass ? <EyeOff size={20} /> : <Eye size={20} />}
                  </button>
                </div>
              </div>
            )}

            {error && (
                <div className="flex items-center gap-2 text-destructive text-sm px-2 animate-in fade-in slide-in-from-top-1">
                    <AlertCircle size={16} />
                    <span>{error}</span>
                </div>
            )}

            <div className="flex items-center justify-between pt-8">
              <Link href="/login" className="text-sidebar-primary font-semibold text-sm hover:underline">
                Back to Login
              </Link>
              <button type="submit" className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:shadow-lg transition-all">
                {step === 3 ? "Update Password" : "Next"}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}